/*---------------------------------------------------------------------------*\

    FILE....: PLAYREC.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 9/2/98

    This module implements the play and record functions for the VPB API.
	 
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

extern "C" {
#include "alawmulaw.h"
}
#include "apifunc.h"
#include "vpbapi.h"
#include "comm.h"
#include "objtrack.h"
#include "config.h"
#include "hostdsp.h"
#include "mapdev.h"
#include "playrec.h"
#include "wavecpp.h"
#include "generic.h"
#include "v4logagc.h"
#include "mess.h"

#include <assert.h>
#include <memory>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <ctype.h>
#include <math.h>						

/*---------------------------------------------------------------------------*\

				    DEFINES

\*---------------------------------------------------------------------------*/

// play/record states

#define IDLE           0
#define PLAYING        1
#define RECORDING      2

// mode flags

#define PLAYREC_WAV    0
#define PLAYREC_VOX    1
#define PLAYREC_SYNC   0
#define PLAYREC_ASYNC  1

// terminate digits

#define	MAX_TERM	31	// maximum number of terminate digits
#define	NUM_VALID_DIG	16	// number of valid digits

// filter
#define HP_ORDER        3       // 2 tap high pass filter

// used to pass arguments to play_file_async_thread via void ptr

typedef struct {
	int     handle;
	void	*wout;
	FILE	*vout;
	char	*buf;
	USHORT	bytesread;
	USHORT	bytes;
	int	data;
        int     file_type;
} PLAY_FILE_ASYNC_ARGS;

// used to pass arguments to record_file_async_thread via void ptr

typedef struct {
	int	handle;
	void	*win;
	FILE	*vin;
	char	*buf;
	USHORT	bytes;
	int	data;
        int     file_type;
} RECORD_FILE_ASYNC_ARGS;

// used to pass arguments to record_file_async_thread via void ptr

typedef struct {
	int	handle;
	ULONG	size;
	char	*masterbuf;
	char	*buf;
	USHORT	bytes;
	int	data;
} RECORD_BUF_ASYNC_ARGS;

typedef struct {
    char   term_digits[MAX_TERM+1]; // string of digits to terminate collection
    USHORT mode;		    // current play mode
    USHORT buf_underflow;	    // non-zero if buf has underflowed
    USHORT underflow_valid;	    // non-zero if underflow valid (in some 
	                            // conditions we expect underflows)
    int    terminate;               // true if we should terminate early     
    int    state;                   // IDLE || PLAYING
    float  total_gain;              // Total gain 
    float  hw_gain;                 // hardware gain parameter
    float  sw_gain;                 // software gain parameter
    float  linear_gain;             // linear gain
    float  hp[HP_ORDER-1];          // HP filter states	
} PLAY;

typedef struct {
    char  term_digits[MAX_TERM+1];  // string of digits to terminate collection
    USHORT mode;		    // current record mode
    unsigned int time_out;
    unsigned int time_start;
    USHORT       buf_overflow;	    // non-zero if buf has overlowed
    int          terminate;         // true if we should terminate early     
    int          state;             // IDLE || RECORDING
    float        total_gain;        // Total gain
    float        hw_gain;           // hardware gain parameter
    float        sw_gain;           // software gain parameter            
    float        linear_gain;       // Linear gain 
    void         *v4log_agcstate;   // AGC state (V4LOG card only)
    float        hp[HP_ORDER-1];    // HP filter states	
} RECORD;

/*--------------------------------------------------------------------------*\

				 STATICS

\*--------------------------------------------------------------------------*/

// states of play and record buf play functions

static PLAY        *play;	// play params
static RECORD      *record;	// record params
static int         sleepms=20;	// time in ms to sleep between polling DSP

static float       words2bytes[] = {2,1,1,0.5};
static float       bytes2words[] = {0.5,1,1,2};

// valid terminate digits

static char valid_dig[] = "0123456789*#ABCD";

// diagnostics

static USHORT	fplay_min;	// minimum words in DSP play (write) buffer
static USHORT	frec_max;	// maximum words in DSP record (read) buffer

static int v4numch;		//for playrec close of v4agc

// hp filter coeffs, 3dB gain at 2000Hz, 9dB at 3000Hz 
static float hp_coeff[] = { 2.00000,  -1.41421,   0.50000};  

/*---------------------------------------------------------------------------*\

			    LOCAL FUNCTION HEADERS

\*---------------------------------------------------------------------------*/

void record_file(int handle, char file_name[], unsigned short mode);
void playloop(int handle, char *buf, USHORT bytesread, USHORT bytes, 
	      void *wout, FILE *vout, int file_type);
void recordloop(int handle, char *buf, USHORT bytes, void *win, FILE *vin, 
		int file_type);
void play_file_async_thread(void *pvargs);
void record_file_async_thread(void *args);
void record_configure_VPB4(int handle, USHORT b, USHORT mode);
void record_configure_VPB8L(int handle, USHORT b, USHORT mode);
static void validate_digits(char *digits);
static int digit_match(char digit, char *term_digits);
void play_voxfile(int handle, char file_name[], USHORT mode);
void record_voxfile(int handle, char file_name[], unsigned short mode);
int time_out(int handle);
void start_time_out(int handle);
void play_set_hw_gain(int handle, float gain, Comm *c);
void record_set_hw_gain(int handle, float gain, Comm *c); 
void set_codec_reg(int    chdev, // channel handle
		   USHORT addr,  // 8-bit address of codec register
		   USHORT data,  // 8-bit data to write to register
		   Comm   *c);
int gain_vector(float g, short *v, int n);
void hp_filter(float states[HP_ORDER], float coeff[HP_ORDER],  
	      short *v, int n);

/*---------------------------------------------------------------------------*\

			       FUNCTIONS

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_open
	AUTHOR..: David Rowe
	DATE....: 9/3/98

	Initialises the playrec module.

\*---------------------------------------------------------------------------*/

void playrec_open(USHORT numch) {
	play = new PLAY[numch];	
	record = new RECORD[numch];
        v4numch = numch;              // used by playrec_close
	
	int i; 
	for(i=0; i<numch; i++) {

		play[i].term_digits[0] = 0;
		play[i].buf_underflow = 0;
		play[i].underflow_valid = 0;
		play[i].mode = VPB_LINEAR;
		play[i].terminate = 0;
		play[i].state = IDLE;
                play[i].sw_gain = 0;            /* JK 9/4/02 */
                play[i].hp[0] = 0.0;
                play[i].hp[1] = 0.0;

		record[i].term_digits[0] = 0;
		record[i].time_out = 0;
		record[i].time_start = 0;
		record[i].buf_overflow = 0;
		record[i].mode = VPB_LINEAR;
		record[i].terminate = 0;
		record[i].state = IDLE;
                record[i].sw_gain = 0;           /* JK 9/4/02 */  
                record[i].hp[0] = 0.0;
                record[i].hp[1] = 0.0;

		// set default gains
		
		if ((model == VPB4) || (vpbreg_get_model() == VPB_V12PCI)) {
			VPBREG *vr = vpb_c->vpbreg(0);

		        play[i].hw_gain = vr->defPlayGain;
			record[i].hw_gain = vr->defRecordGain;
			play[i].sw_gain = 0;
			record[i].sw_gain = 0;
			play[i].total_gain = play[i].hw_gain + play[i].sw_gain;
			record[i].total_gain = record[i].hw_gain + record[i].sw_gain;

			play_set_hw_gain(i, play[i].hw_gain, vpb_c);
			record_set_hw_gain(i, record[i].hw_gain, vpb_c);
			
			// set linear gain factor
			play[i].linear_gain = pow(10.0,(play[i].sw_gain)/20.0);
			record[i].linear_gain = pow(10.0,(record[i].sw_gain)/20.0);

			if (vpbreg_get_model() == VPB_V4PCI) {
				if (!vpb_c->vpbreg(0)->hybdefs_overwrite) {
					// defaults havent been overwritten
					// so set them here based on version
					// we have to set defs here as eeprom isnt
					// read until after vpbreg code is
					// executed

					float revision;
					revision = atof(vpb_c->vpbreg(0)->revision);
					if (revision > 19.0) {
						mprintf("playrec.cpp: setting V19 & > hyb bals\n");
						vpb_c->vpbreg(0)->defbal1 = 0xec;
						vpb_c->vpbreg(0)->defbal2 = 0x70;
						vpb_c->vpbreg(0)->defbal3 = 0xf1;
					}
				}
				// otherwise use defaults set in vpbreg.cpp at boot
				// time, e.g. hard coded or env variable-set values

				// default hybrid balance (V4PCI only)
				set_codec_reg(i,0x32,vpb_c->vpbreg(0)->defbal1, vpb_c);
				set_codec_reg(i,0x3a,vpb_c->vpbreg(0)->defbal2, vpb_c);
				set_codec_reg(i,0x42,vpb_c->vpbreg(0)->defbal3, vpb_c);

			}

			if (vpbreg_get_model() == VPB_V4LOG) {
				// default hybrid balance (V4PCI only)
				set_codec_reg(i, 0x32, 199, vpb_c);
				// logging card AGC
                                v4log_agc_open(&record[i].v4log_agcstate);

			}
		}		
	}

	fplay_min = 64000;
	frec_max = 0;

}

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_close
	AUTHOR..: David Rowe
	DATE....: 9/3/98

	Closes the playrec module.

\*--------------------------------------------------------------------------*/

void playrec_close() {

        int  i;  
	if (vpbreg_get_model() == VPB_V4LOG) {
            for(i=0; i<v4numch; i++) {  
                v4log_agc_close(record[i].v4log_agcstate);
	   }
        }

	delete [] play;
	delete [] record;
}

/*---------------------------------------------------------------------------*\

			       PLAY FUNCTIONS

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: play_buf_start
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Prepares a channel for playing a buffer by configuring the channel for
	the appropriate compression rate.  Internal API version.

\*--------------------------------------------------------------------------*/

void play_buf_start(int handle, unsigned short mode)
//	int		handle		handle of destination device
//	unsigned short	mode		specifies compression mode
{
	ValidHandleCheck(handle);
	play[handle].mode = mode;
	play[handle].state = PLAYING;
        play[handle].terminate = 0;
	if (vpbreg_get_model() == VPB_V12PCI) {
//		#ifndef WIN32
		#ifdef LINUX
		HostDSPSetupPlay(handle);
		#endif
	}
}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_buf_sync
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Plays a buffer of speech samples to a VPB channel.  Returns when all of
	the buffer has been sent to the VPB's DSP FIFO, ie operates 
	synchronously.  Note that when this function exits the DSP is still 
	playing info, use the finish function below to wait for the DSP FIFO
	to empty.

	If another buffer is to be sent, then when this function exits
	there is plenty of time to prepare the next buffer before the DSP fifo
	empties.

\*--------------------------------------------------------------------------*/

int play_buf_sync(int handle, char *buf, unsigned short length)
//	int		handle		handle of destination device
//	char		*buf		ptr to buffer to send
//	unsigned short	length		length of buffer in bytes
{
	USHORT		b,ch;		// board and channel for this handle
	word		*wordbuf;	// buffer of wrods to download
	USHORT		words;	        // num of samples (words) to download
	USHORT		bytes;		// number of bytes to read from buf
	VPBREG		*v;		// ptr to vpbreg for this card
	char            *tmp;

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	v = vpb_c->vpbreg(b);
	vpb_c->CheckForAssert(b);
	
	words = v->sztxdf[ch]/2;
	wordbuf = new word[v->sztxdf[ch]];
	tmp = new char[v->sztxdf[ch]];
	int f;

	while(length && !play[handle].terminate) {

	  // wait while DSP fifo is full

	  while ( (words = v->txdf[ch]->HowEmpty()) == 0)  	
	    GenericSleep(sleepms);

	  // determine how empty buf gets for diagnostic purposes

	  f = v->txdf[ch]->HowFull();
	  if (f < fplay_min) {
	    fplay_min = f;
	  }
	  
	  // convert buf samples to linear

	  bytes = (USHORT)(words*words2bytes[play[handle].mode]);
	  if (bytes > length)  {
	    bytes = length;
	    words = (USHORT)(bytes*bytes2words[play[handle].mode]);
	  }
	  
	  switch(play[handle].mode) {
	  case VPB_LINEAR:
	    memcpy(wordbuf, buf, bytes);
	    break;
	  case VPB_ALAW:
	    alaw_decode((short*)wordbuf, buf, words);  
	    break;
	  case VPB_MULAW:
	    mulaw_decode((short*)wordbuf, buf, words);  
	    break;
	  case VPB_OKIADPCM:
	    assert(0); // sorry - not supported just yet!
	    break;
	  case VPB_OKIADPCM24:
	    assert(0); // sorry - not supported just yet!
	    break;
	  default:
	    assert(0);
	  }

	  // insert software gain
	  gain_vector(play[handle].linear_gain,(short *)wordbuf,
		      words);
	 
	  // HP filter
	  if (vpb_c->vpbreg(0)->hp_enable) {
		  hp_filter(play[handle].hp, hp_coeff, (short*)buf, 
			    (bytes/sizeof(short)));
	  }
	    	  
	  // alaw compand here to prevent clipping bug
       	  alaw_encode(tmp, (short*)wordbuf, words);
	  alaw_decode((short*)wordbuf, tmp, words);
	  
	  length -= bytes;
	  buf += bytes;

	  // send to DSP tx FIFO

	  if (v->txdf[ch]->Write(wordbuf, words) != 0)
	    assert(0);

	}

	delete [] wordbuf;
	delete [] tmp;

	if (play[handle].terminate) {
		return(VPB_FINISH);
	}
	else
		return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_buf_finish
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Completes the playing of a channel.

\*--------------------------------------------------------------------------*/

void play_buf_finish(int handle)
//	int     handle		handle of destination device
{
	USHORT	b,ch;		// board and channel for this handle
	VPBREG	*v;		// ptr to vpbreg for this card

	// validate

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	v = vpb_c->vpbreg(b);
		
	// stop checking for underflows

	play[handle].underflow_valid = 0;

	// wait for DSP FIFO to empty
	
	while (v->txdf[ch]->DspHowFull() >= v->lsf) {
		GenericSleep(sleepms);
	}
	play[handle].state = IDLE;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_file
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to play a file to a channel.  This function is used
	to create the async and sync play functions, and handles wave and
	vox file formats.

\*--------------------------------------------------------------------------*/

void play_file(int handle, char file_name[], USHORT mode, int file_type, 
	       int sync, int data)
//	int	 handle		handle of destination device
//	char	 file_name[]	name of file to play
//      USHORT   mode           mode (only required for PLAYREC_VOX)
//      int      file_type      PLAYREC_WAVE || PLAYRECVOX
//      int      sync           PLAYREC_SYNC || PLAYREC_ASYNC
//	int	 data	        VPB_PLAYEND event data (only PLAYREC_ASYNC)
{
	void	 *wout;		// wave file
	FILE	 *vout;	        // vox file
	USHORT	 words;		// words to download to FIFO
	USHORT	 bytes;		// num bytes to try to read from file
	USHORT	 bytesread;	// num bytes read from file
	char	 *buf;		// buffer of bytes read from file
	USHORT	 b,ch;		// board and channel for this handle

	// Validation ----------------------------------------------------

	mprintf("play_file: starting\n");
	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	if (file_type == PLAYREC_WAV) {
	  mprintf("play_file: opening wave file\n");
	  wave_open_read(&wout, file_name);
	  wave_get_mode(wout, &mode);
	}
	else {
	  mprintf("play_file: opening vox file\n");
	  vout = fopen(file_name,"rb");
	  if (vout == NULL) 
	    throw Wobbly(VPBAPI_PLAY_CANT_OPEN_VOXFILE);
	}

	mprintf("play_file: just before play_buf_start\n");
	play_buf_start(handle, mode);

	// calculate buffer size to fit DSP FIFO

	words = vpb_c->vpbreg(b)->sztxdf[ch]/2;
	bytes = (USHORT)(words*words2bytes[play[handle].mode]);
	buf = new char[bytes];
		
	// initial read

	mprintf("play_file: just before initial read\n");
	if (file_type == PLAYREC_WAV) 
	  bytesread = vpb_wave_read(wout, buf, bytes);
	else 
	  bytesread = fread(buf, sizeof(char), bytes, vout);

	// main play loop

	if (sync == PLAYREC_SYNC) {
	  playloop(handle, buf, bytesread, bytes, wout, vout, file_type);
	}
	else {
	  PLAY_FILE_ASYNC_ARGS *args = new PLAY_FILE_ASYNC_ARGS;
	  args->handle = handle;
	  args->wout = wout;
	  args->vout = vout;
	  args->buf = buf;
	  args->bytesread = bytesread;
	  args->bytes = bytes;
	  args->data = data;
	  args->file_type = file_type;

	  mprintf("play_file: just before begin thread\n");
	  Generic_beginthread(play_file_async_thread, 0, (void*)args);
	}
}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_file_async_thread
	AUTHOR..: David Rowe
	DATE....: 23/3/98

	Handles main loop for async play functions. 

\*---------------------------------------------------------------------------*/

void play_file_async_thread(void *pvargs)
//	void	*args;		arguments structure
{
	PLAY_FILE_ASYNC_ARGS *args = (PLAY_FILE_ASYNC_ARGS*)pvargs;
	int		     handle = args->handle;
	void	             *wout = args->wout;
	FILE	             *vout = args->vout;
	char	             *buf = args->buf;
	USHORT	             bytesread = args->bytesread;
	USHORT	             bytes = args->bytes;
	int		     data = args->data;
	int		     file_type = args->file_type;
	VPB_EVENT	     e;

	delete args;
	
	mprintf("play_file_async_thread starting\n");
	GenericSetThreadPriorityHigh();

	try {
	  playloop(handle, buf, bytesread, bytes, wout, vout, file_type);
	  mprintf("play_file_async_thread: just after playloop\n");

	  e.type = VPB_PLAYEND;
	  e.handle = handle;
	  e.data = data;

	  play_buf_finish(handle);
	  mprintf("play_file_async_thread: about to post PLAY_END event\n");
	  putevt(&e, 0);		
	  mprintf("play_file_async_thread: after posting PLAY_END event\n");
	}

	// need to handle exceptions here as we can't throw across threads

	catch(Wobbly w) {
		mprintf("Oops - play_file_async_thread bombed!\n");

		if (file_type == PLAYREC_WAV) 
		  vpb_wave_close_read(wout);
		else
		  fclose(vout);

		e.type = VPB_PLAYEND;
		e.handle = handle;
		e.data = w.code;
		try {
		  play_buf_finish(handle);
		}
		catch(Wobbly w){
			mprintf("Oops - play_file_async_thread bombed "
				"again!\n");
		}

		putevt(&e, 0);		
	}
	Generic_endthread();
	mprintf("play_file_async_thread ended normally\n");
}

/*---------------------------------------------------------------------------*\

	FUNCTION: playloop
	AUTHOR..: David Rowe
	DATE....: 10/3/02

	Helper function containing common code for main loop of sync and async
	play functions.

\*---------------------------------------------------------------------------*/

void playloop(int handle, char *buf, USHORT bytesread, USHORT bytes, 
	      void *wout, FILE *vout, int file_type)
{
  while(bytesread && !play[handle].terminate) {
    play_buf_sync(handle, buf, bytesread);
    if (file_type == PLAYREC_WAV) 
      bytesread = vpb_wave_read(wout, buf, bytes);
    else 
      bytesread = fread(buf, sizeof(char), bytes, vout);
  }

  delete [] buf;

  if (file_type == PLAYREC_WAV) 
    vpb_wave_close_read(wout);
  else
    fclose(vout);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_new_digit_play
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Called by the MMQ when DSP detects a digit event.  Causes the play
	operation on the current channel to be terminated.

\*---------------------------------------------------------------------------*/

void playrec_new_digit_play(int h, USHORT digit) {
	if ((play[h].state == PLAYING) && 
	    digit_match((char)digit, play[h].term_digits)) {
	  play[h].terminate = 1;
	}
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_set
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Sets play parameters or this channel.

\*---------------------------------------------------------------------------*/

int WINAPI vpb_play_set(int handle, VPB_PLAY *vpb_play)
{
	try {
		ValidHandleCheck(handle);
		validate_digits(vpb_play->term_digits);
		strcpy(play[handle].term_digits,vpb_play->term_digits);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_set"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_buf_start
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	API callable version of play_buf_start.

\*---------------------------------------------------------------------------*/

int WINAPI vpb_play_buf_start(int handle, unsigned short mode)
//	int	        handle		handle of destination device
//	unsigned short	mode		specifies compression mode
{

	try {
		play_buf_start(handle, mode);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_buf_start"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_buf_sync
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	API version of play_buf_sync().  Returns VPB_FINISH if playing
	should finish, else returns VPB_OK.

\*---------------------------------------------------------------------------*/

int WINAPI vpb_play_buf_sync(int handle, char *buf, unsigned short length)
//	int		handle		handle of destination device
//	char		*buf		ptr to buffer to send
//	unsigned short	length		length of buffer in bytes
{
	int		ret;

	try {
		ret = play_buf_sync(handle, buf, length);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_buf_sync"));
	}
	
	return(ret);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_buf_finish
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Completes the playing of a channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_buf_finish(int handle)
//	int	handle		handle of destination device
{
	try {
		play_buf_finish(handle);	
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_buf_finish"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_file_sync
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to play a file to a channel.  Function returns when
	playing is finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_file_sync(int handle, char file_name[])
//	int	 handle		handle of destination device
//	char	 file_name[]	name of file to play
{

	try {
		play_file(handle, file_name, 0, PLAYREC_WAV, PLAYREC_SYNC, 0);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_file_sync"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_voxfile_sync
	AUTHOR..: David Rowe & john kostogiannis
	DATE....: 20/8/98

	Function to play a vox file to a channel.  Function returns when
	playing is finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_voxfile_sync(int handle, char file_name[], 
				 unsigned short mode)
//	int		handle		handle of destination device
//	char		file_name[]	name of file to play
//	unsined short	mode		compression mode
{

	try {
	  play_file(handle, file_name, (int)mode, PLAYREC_VOX, PLAYREC_SYNC,0);
	}

	catch(Wobbly w){
	  return(RunTimeError(w,"vpb_play_voxfile_sync"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_file_async
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to play a file to a channel.  Function returns as soon
	as playing has started, and places an event on the API Q when playing
	has finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_file_async(int handle, char file_name[], int data)
//	int	handle		handle of destination device
//	char	file_name[]	name of file to play
//	int	data		VPB_PLAYEND event data
{

	try {
	  play_file(handle, file_name, 0, PLAYREC_WAV, PLAYREC_ASYNC,
		    data);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_file_async"));
	}
	
	return(VPB_OK);
}
		
/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_voxfile_async
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to play a vox file to a channel.  Function returns as
	soon as playing has started, and places an event on the API Q when 
	playing has finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_voxfile_async(int handle, char file_name[], 
				  unsigned short mode,int data)
//	int	    handle	handle of destination device
//	char	    file_name[]	name of file to play
//	USHORT	    mode	compression mode
//	int	    data	VPB_PLAYEND event data
{

	try {
	  play_file(handle, file_name, (int)mode, PLAYREC_VOX, PLAYREC_ASYNC,
		    data);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_voxfile_async"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_terminate
	AUTHOR..: David Rowe
	DATE....: 24/3/98

	Stops playing immediately. Does nothing if we are not playing.
	NOTE: this doesnt have to be in the same thread context as 
	vpb_play_buf_xxxx
	
\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_terminate(int handle)
//	int	handle	handle of channel 
{
	try {
		ValidHandleCheck(handle);
		play[handle].terminate = 1;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_terminate"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

			     RECORD FUNCTIONS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: record_buf_start
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Prepares a channel for recording a buffer by configuring the channel 
	forthe appropriate compression rate.  Internal API version.

\*--------------------------------------------------------------------------*/

void record_buf_start(int handle, unsigned short mode)
//	int		handle		handle of destination device
//	unsigned short	mode		specifies compression mode
{
	USHORT		b,ch;		// board and channel for this handle
	USHORT		words;
	word		*wordbuf;
	VPBREG		*v;		

	ValidHandleCheck(handle);
	record[handle].mode = mode;
        record[handle].terminate = 0;
	record[handle].state = RECORDING;
	start_time_out(handle);

	// empty record FIFO

	maphndletodev(handle, &b, &ch);
	v = vpb_c->vpbreg(b);
        words = v->rxdf[ch]->HowFull();
	wordbuf = new word[words];
	v->rxdf[ch]->Read(wordbuf, words);
	delete [] wordbuf;	
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_buf_sync
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Records a buffer of speech samples from a VPB channel.  Returns when
	the
	buffer if filled from the VPB's DSP FIFO, ie operates synchronously.  
	Note that when this function exits the DSP is still recording info, the
	finish function must be called to disable recording.

	If another buffer is to be obtained, then when this function exits
	there is plenty of time to prepare for the next buffer before the DSP
	fifo fills.

\*--------------------------------------------------------------------------*/

int record_buf_sync(int handle, char *buf, unsigned short length)
//	int		handle		handle of destination device
//	unsigned short	mode		specifies compression mode and async/sync
//	char		*buf		ptr to buffer to receive
//	unsigned short	length		length of buffer in bytes
{
	USHORT		b,ch;		// board and channel for this handle
	word		*wordbuf;	// buffer of wrods to download
        word            *agcbuf;	// buffer of words for agc
 	USHORT		words;		// number of samples (words) to upload
	USHORT		bytes;		// number of bytes to read from buf
	VPBREG		*v;		// ptr to vpbreg for this card

	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	v = vpb_c->vpbreg(b);
		
	// upload half DSP FIFO size buffers at a time

	words = v->sztxdf[ch]/2;
	wordbuf = new word[words];
        agcbuf = new word[words]; 	         

	USHORT f;
	while(length && !record[handle].terminate && !time_out(handle)) {

	  // wait for buffer to half fill

	  while ((f = v->rxdf[ch]->HowFull()) < words) {
	    GenericSleep(sleepms);
	  }
	  if (f > frec_max)
	    frec_max = f;

	  // convert buf samples to linear

	  bytes = (USHORT)(words*words2bytes[record[handle].mode]);
	  if (bytes > length)  {
	    bytes = length;
	    words = (USHORT)(bytes*bytes2words[record[handle].mode]);
	  }
	  if (v->rxdf[ch]->Read(wordbuf, words) != 0)
	    assert(0);		

          // switch in agc for V4LOG card 
          if (vpbreg_get_model() == VPB_V4LOG) {
                  v4log_agc_apply(record[handle].v4log_agcstate,              
                            (short*)agcbuf,(short*)wordbuf, words);
                  memcpy(wordbuf, agcbuf, bytes);
          }

	  // insert software gain	  
	  	     
          gain_vector(record[handle].linear_gain,
                (short*)wordbuf,(bytes/sizeof(short))); 	     

	  // HP filter
	  if (vpb_c->vpbreg(0)->hp_enable) {
		  hp_filter(record[handle].hp, hp_coeff, (short*)wordbuf, 
			    (bytes/sizeof(short)));
	  }

	  switch(record[handle].mode) {
	  case VPB_LINEAR:
	    memcpy(buf, wordbuf, bytes);
	    break;
	  case VPB_ALAW:
	    alaw_encode(buf, (short*)wordbuf, words);  
	    break;
	  case VPB_MULAW:
	    mulaw_encode(buf, (short*)wordbuf, words);  
	    break;
	  case VPB_OKIADPCM:
	    assert(0); // sorry - not supported just yet!
	    break;
	  case VPB_OKIADPCM24:
	    assert(0); // sorry - not supported just yet!
	    break;
	  default:
	    assert(0);
	  }

	  length -= bytes;
	  buf += bytes;
	}

	delete [] wordbuf;
        delete []agcbuf;           

	if (record[handle].terminate || time_out(handle)) {
	  return(VPB_FINISH);
	}
	else
	  return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_buf_finish
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Completes the recording of a channel.

\*--------------------------------------------------------------------------*/

void record_buf_finish(int handle)
//	int    handle		handle of source device
{
  record[handle].state = IDLE; 
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_file
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to record a file from a channel.  This function is 
	used to create the async and sync record functions, and handles wave 
	and vox file formats.  The function continues until the recording is 
	terminated.

\*--------------------------------------------------------------------------*/

void record_file(int handle, char file_name[], USHORT mode, int file_type, 
	       int sync, int data)
//	int	 handle		handle of source device
//	char	 file_name[]	name of file to record
//	USHORT	 mode		specifies compression mode
//      int      file_type      PLAYREC_WAVE || PLAYRECVOX
//      int      sync           PLAYREC_SYNC || PLAYREC_ASYNC
//	int	 data	        VPB_PLAYEND event data (only PLAYREC_ASYNC)
{
	void		*win;
	FILE            *vin;
	USHORT		words;		// words to upload from FIFO
	USHORT		bytes;		// num bytes to try to read from file
	char		*buf;		// buffer of bytes read from file
	USHORT		b,ch;		// board and channel for this handle
	
	// Validation ----------------------------------------------------

	ValidHandleCheck(handle);
	maphndletodev(handle, &b, &ch);
	if (file_type == PLAYREC_WAV) {
	  wave_open_write(&win, file_name, mode);
	}
	else {
	  vin = fopen(file_name,"wb");
	  if (vin == NULL) 
	    throw Wobbly(VPBAPI_RECORD_CANT_OPEN_VOXFILE);
	}

	record_buf_start(handle, mode);

	// calculate buffer size to fit DSP FIFO

	words = vpb_c->vpbreg(b)->szrxdf[ch]/2;
	bytes = (USHORT)(words*words2bytes[record[handle].mode]);
	buf = new char[bytes];
		
	// main record loop

	if (sync == PLAYREC_SYNC) {
	  recordloop(handle, buf, bytes, win, vin, file_type);
	}
	else {
	  RECORD_FILE_ASYNC_ARGS *args = new RECORD_FILE_ASYNC_ARGS;
	  args->handle = handle;
	  args->win = win;
	  args->vin = vin;
	  args->buf = buf;
	  args->bytes = bytes;
	  args->file_type = file_type;

	  Generic_beginthread(record_file_async_thread, 0, (void*)args);
	}
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_file_async_thread
	AUTHOR..: David Rowe
	DATE....: 23/3/98

	Handles main loop for async record functions. 

\*---------------------------------------------------------------------------*/

void record_file_async_thread(void *pvargs)
//	void	*args;		arguments structure
{
	RECORD_FILE_ASYNC_ARGS *args = (RECORD_FILE_ASYNC_ARGS*)pvargs;
	int		     handle = args->handle;
	void	             *win = args->win;
	FILE	             *vin = args->vin;
	char	             *buf = args->buf;
	USHORT	             bytes = args->bytes;
	int		     file_type = args->file_type;
	VPB_EVENT	     e;

	delete args;
	
	GenericSetThreadPriorityHigh();

	try {
	  recordloop(handle, buf, bytes, win, vin, file_type);

	  if(time_out(handle))
	    e.data = VPB_RECORD_TIMEOUT;

	  e.type = VPB_RECORDEND;
	  e.handle = handle;

	  record_buf_finish(handle);
	  putevt(&e, 0);		
	}

	// need to handle exceptions here as we can't throw across threads

	catch(Wobbly w) {

		if (file_type == PLAYREC_WAV) 
		  vpb_wave_close_write(win);
		else
		  fclose(vin);

		e.type = VPB_RECORDEND;
		e.handle = handle;
		e.data = w.code;
		try {
		  record_buf_finish(handle);
		}
		catch(Wobbly w){}

		putevt(&e, 0);		
	}
	Generic_endthread();
}

/*---------------------------------------------------------------------------*\

	FUNCTION: recordloop
	AUTHOR..: David Rowe
	DATE....: 10/3/02

	Helper function containing common code for main loop of sync and async
	record functions.

\*---------------------------------------------------------------------------*/

void recordloop(int handle, char *buf, USHORT bytes, void *win, FILE *vin, 
		int file_type)
{
  USHORT byteswrite;

  while(!record[handle].terminate && !time_out(handle)) {
    record_buf_sync(handle, buf, bytes);
    if (file_type == PLAYREC_WAV) 
      byteswrite = vpb_wave_write(win, buf, bytes);
    else 
      byteswrite = fwrite(buf, sizeof(char), bytes, vin);
  }

  delete [] buf;

  record_buf_finish(handle);
  if (file_type == PLAYREC_WAV) {
    vpb_wave_close_write(win);
  }
  else
    fclose(vin);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_set
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Sets record parameters or this channel.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_set(int handle, VPB_RECORD *vpb_record)
{
	try {
		ValidHandleCheck(handle);
		validate_digits(vpb_record->term_digits);
		strcpy(record[handle].term_digits,vpb_record->term_digits);
		record[handle].time_out = vpb_record->time_out;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_set"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_buf_start
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	API callable version of record_buf_start.
	NOTE: All vpb_record_buf_xxxx functions for a given channel must be
	executed in the same thread context.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_buf_start(int handle, unsigned short mode)
//	int		handle		handle of destination device
//	unsigned short	mode		specifies compression mode
{

	try {
		record_buf_start(handle, mode);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_buf_start"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_buf_sync
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	API version of record_buf_sync().
	NOTE: All vpb_record_buf_xxxx functions for a given channel must be
	executed in the same thread context.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_buf_sync(int handle, char *buf, unsigned short length)
//	int	        handle		handle of destination device
//	char		*buf		ptr to buffer to receive
//	unsigned short	length		length of buffer in bytes
{
	int	ret;

	try {
		ret = record_buf_sync(handle, buf, length);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_buf_sync"));
	}
	
	return(ret);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_buf_finish
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Completes the recording of a channel.
	NOTE: All vpb_record_buf_xxxx functions for a given channel must be
	executed in the same thread context.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_buf_finish(int handle)
//	int	handle		handle of destination device
{
	try {
		record_buf_finish(handle);	
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_buf_finish"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_file_sync
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to record a file to a channel.  Function returns when
	recording is finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_file_sync(int handle, char file_name[], unsigned short mode)
//	int		handle		handle of destination device
//	char	        file_name[]	name of file to record
//	unsigned short	mode		specifies compression mode
{

	try {
	  record_file(handle, file_name, mode, PLAYREC_WAV, PLAYREC_SYNC, 0);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_file_sync"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_voxfile_sync
	AUTHOR..: David Rowe & John Kostogiannis
	DATE....: 20/8/98

	Utility function to record a vox file to a channel.  Function returns 
	when recording is finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_voxfile_sync(int handle, char file_name[], unsigned short mode)
//	int		handle		handle of destination device
//	char		file_name[]	name of file to record
//	unsigned short	mode		specifies compression mode
{

	try {
	  record_file(handle, file_name, mode, PLAYREC_VOX, PLAYREC_SYNC, 0);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_voxfile_sync"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_file_async
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to record a file from a channel.  Function returns as 
	soon as recording has started, and places an event on the API Q when 
	recording has finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_file_async(int handle, char file_name[], 
				 unsigned short mode)
//	int		handle		handle of destination device
//	char		file_name[]	name of file to record
//	unsigned short	mode		specifies compression mode
//	int		data		VPB_RECORDEND event data
{
	try {
	  record_file(handle, file_name, mode, PLAYREC_WAV, PLAYREC_ASYNC, 0);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_file_async"));
	}
	
	return(VPB_OK);
}
		
/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_voxfile_async
	AUTHOR..: David Rowe
	DATE....: 20/3/98

	Utility function to record a vox file from a channel.  Function 
	returns as soon as recording has started, and places an event on the 
	API Q when recording has finished.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_voxfile_async(int handle, char file_name[], 
				    unsigned short mode)
//	int		handle		handle of destination device
//	char	        file_name[]	name of file to record
//	unsigned short	mode		specifies compression mode
//	int		data		VPB_RECORDEND event data
{
	try {
	  record_file(handle, file_name, mode, PLAYREC_VOX, PLAYREC_ASYNC, 0);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_voxfile_async"));
	}
	
	return(VPB_OK);
}
		
/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_terminate
	AUTHOR..: David Rowe
	DATE....: 24/3/98

	Stops recording immediately. Does nothing if we are not recording.
	NOTE: this doesnt have to be in the same thread context as 
	vpb_record_buf_xxxx
	
\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_terminate(int handle)
//	int	handle	handle of channel 
{
	try {
		record[handle].terminate = 1;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_terminate"));
	}
	
	return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

			   DIGIT TERMINATING FUNCTIONS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: validate_digits
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Determines if the digit string contains valid characters and is of
	valid length.

\*--------------------------------------------------------------------------*/

static void validate_digits(char *digits)
{
	int i,j;
	
	// validate this entries digit string

	USHORT len = strlen(digits);
	if (len > MAX_TERM)
		throw Wobbly(VPBAPI_TERM_DIGITS_STRING_TOO_LONG);

	for(j=0; j<len; j++) {
		// search for digit in table

		char c = toupper(digits[j]);
		int num = -1;
		for(i=0; i<NUM_VALID_DIG; i++) {
			if (valid_dig[i] == c)
				num = i;
		}
		if (num < 0)
			throw Wobbly(VPBAPI_INVALID_TERM_DIGITS);
	}
}

/*---------------------------------------------------------------------------*\

	FUNCTION: digit_match
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Determines if the digit is a member of the term_digit string.

\*--------------------------------------------------------------------------*/

static int digit_match(char digit, char *term_digits)
{
	char *p = term_digits;

	while(*p != 0) {
		if (*p == digit)
			return(1);
		p++;
	}

	return(0);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_new_digit_record
	AUTHOR..: David Rowe
	DATE....: 29/7/98

	Called by the MMQ when DSP detects a digit event.  Causes the record
	operation on the current channel to be terminated.

\*--------------------------------------------------------------------------*/

void playrec_new_digit_record(int h, USHORT digit) {
	if ((record[h].state == RECORDING) && 
	    digit_match((char)digit, record[h].term_digits)) {
		record[h].terminate = 1;
	}
}

/*--------------------------------------------------------------------------*\

			   DIAGNOSTICS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_diag
	AUTHOR..: David Rowe
	DATE....: 9/8/98

	Returns playrec diagnostics.

\*--------------------------------------------------------------------------*/

void playrec_diag(USHORT *play, USHORT *rec)
{
	*play = fplay_min;
	*rec = frec_max;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_diag_clear
	AUTHOR..: David Rowe
	DATE....: 9/8/98

	Resets playrec diagnostics.

\*--------------------------------------------------------------------------*/

void playrec_diag_reset()
{
	fplay_min = 64000;
	frec_max = 0;
}

/*--------------------------------------------------------------------------*\

                                   GAIN

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_set_gain
	AUTHOR..: David Rowe
	DATE....: 17/8/98

	Sets the gain of the play channel, using the hardware gain of the
	TS5070 codec on the VPB4.
	JK: 9/4/02 Modified to only set software gain and not the hardware 
	gain of the TS5050 codec.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_set_gain(int handle, float gain) 
{
	try {
		// validate

		ValidHandleCheck(handle);
                                
		// Determine sw_gain and convert gain(db) to a linear gain
		play[handle].sw_gain = gain-play[handle].hw_gain;
		play[handle].linear_gain = pow(10.0,(play[handle].sw_gain)/20.0);
		play[handle].total_gain = gain;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_set_gain"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_set_hw_gain
	AUTHOR..: Jason Spence <jspence@lightconsulting.com>
	DATE....: 17/4/02

    Externally accessible wrapper around play_set_hw_gain.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_set_hw_gain(int handle, float gain) 
{
	try {
		// validate

		ValidHandleCheck(handle);
                                
	    play_set_hw_gain(handle, gain, vpb_c);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_set_hw_gain"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: play_set_hw_gain
	AUTHOR..: David Rowe
	DATE....: 9/02/01

	Internal version: Sets the hardware (TS5070 codec) play gain component.

\*--------------------------------------------------------------------------*/

void play_set_hw_gain(int handle, float gain, Comm *c) 
{
	USHORT	b,ch;			 // board and channel for this handle
	word	mess[PC_LCODEC_DNGAIN];

	// gain is in range of +/- 12dB for TS5070

	if ((gain > 12.0) || (gain < -12.0))
		throw Wobbly(VPBAPI_GAIN_OUT_OF_RANGE);

	// now scale float gain to 8 bit unsigned int
	// each unit is 0.1 dB, 0dB is 0x80

	USHORT scaled_gain = (USHORT)(gain*10.0 + 0x80);

	maphndletodev(handle, &b, &ch);
	mess[0] = PC_LCODEC_DNGAIN;
	mess[1] = PC_CODEC_DNGAIN;
	mess[2] = ch;
	mess[3] = scaled_gain;
	c->PutMessageVPB(b,mess);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_play_get_gain
	AUTHOR..: David Rowe
	DATE....: 17/8/98

	Gets the total gain of the play channel. It comprising of a softare 
	and hardware component (TS5070 codec) on the VPB4/V4PCI.
	
\*--------------------------------------------------------------------------*/

int WINAPI vpb_play_get_gain(int handle, float *gain) 
{

	try {
		// validate

		ValidHandleCheck(handle);
		*gain = play[handle].total_gain;			
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_play_get_gain"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_get_gain
	AUTHOR..: David Rowe
	DATE....: 17/8/98

	Gets the total gain of the play channel. It comprising of a softare 
	and hardware component (TS5070 codec) on the VPB4/V4PCI.
	
\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_get_gain(int handle, float *gain) 
{

	try {
		// validate

		ValidHandleCheck(handle);
		*gain = record[handle].total_gain;			
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_get_gain"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_set_gain
	AUTHOR..: John Kostogiannis
	DATE....: 9/4/02

	Sets the total gain of the record channel. The total gain is comprised
        of a software and hardware component (TS5070 codec). Previously it 
	only comprised of the hardware component. 
	JK 9/4/02 Modified to only set sofware record gain. Disabled 
        hardware configured record gain.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_set_gain(int handle, float gain) 
{
	try {
		// validate

		ValidHandleCheck(handle);

		// Determine sw_gain and convert gain(db) to a linear gain
		record[handle].sw_gain = gain-record[handle].hw_gain;
		record[handle].linear_gain = pow(10.0,(record[handle].sw_gain)/20.0);
		record[handle].total_gain = gain;
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_set_gain"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_record_set_hw_gain
	AUTHOR..: Jason Spence <jspence@lightconsulting.com>
	DATE....: 17/4/02

    Externally accessible wrapper around record_set_hw_gain.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_record_set_hw_gain(int handle, float gain) 
{
	try {
		// validate

		ValidHandleCheck(handle);
                                
	    record_set_hw_gain(handle, gain, vpb_c);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_record_set_hw_gain"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: record_set_hw_gain
	AUTHOR..: David Rowe
	DATE....: 9/2/01

	Internal version. Sets the TS5070 codec record gain.

\*--------------------------------------------------------------------------*/

void record_set_hw_gain(int handle, float gain, Comm *c) 
{
	USHORT	b,ch;	                 // board and channel for this handle
	word	mess[PC_LCODEC_UPGAIN];

	if ((gain > 12.0) || (gain < -12.0))
		throw Wobbly(VPBAPI_GAIN_OUT_OF_RANGE);

	// now scale float gain to 8 bit unsigned int
	// each unit is 0.1 dB, 0dB is 0x80

	USHORT scaled_gain = (USHORT)(-gain*10.0 + 0x80);

	maphndletodev(handle, &b, &ch);
	mess[0] = PC_LCODEC_UPGAIN;
	mess[1] = PC_CODEC_UPGAIN;
	mess[2] = ch;
	mess[3] = scaled_gain;
	c->PutMessageVPB(b,mess);
}

// test functions, DR 12/9/02

void record_get_hw_gain(int handle, float *gain) 
{
	*gain = record[handle].hw_gain;	
}
void record_get_sw_gain(int handle, float *gain) 
{
	*gain = record[handle].sw_gain;	
}

void play_get_hw_gain(int handle, float *gain) 
{
	*gain = play[handle].hw_gain;	
}
void play_get_sw_gain(int handle, float *gain) 
{
	*gain = play[handle].sw_gain;	
}

/*--------------------------------------------------------------------------*\

                                 TIME OUTS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: start_time_out
	AUTHOR..: David Rowe
	DATE....: 21/8/98

	Starts the time out logic for a given record session.

\*--------------------------------------------------------------------------*/

void start_time_out(int handle)
{
	record[handle].time_start = GenerictimeGetTime();
}

/*---------------------------------------------------------------------------*\

	FUNCTION: time_out
	AUTHOR..: David Rowe
	DATE....: 21/8/98

	Returns non-zero if record session has timed out.  Should be called 
	Starts the time out logic for a given record session.

\*--------------------------------------------------------------------------*/

int time_out(int handle)
{
  if (record[handle].time_out) {
    return ((record[handle].time_out + 
	     record[handle].time_start) < (unsigned)GenerictimeGetTime());
  }
  else
    return(0);
}

/*--------------------------------------------------------------------------*\

                                 ALARMS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_reset_record_fifo_alarm
	AUTHOR..: David Rowe
	DATE....: 23/9/98

	Resets the latch that detects record DSP buffer overflows.  After
	calling this function another event may be generated if the buffer
	overflows.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_reset_record_fifo_alarm(int handle)
{
	USHORT	stobj,b,ch;

	try {
		ValidHandleCheck(handle);
		maphndletodev(handle, &b, &ch);
		stobj = objtrack_handle_to_id(UPOBJ, handle);

		switch (vpb_c->vpbreg(b)->model) {
			case VPB4:
				// PACKED_FIFO_UP
				config_reset_fifo_alarm(vpb_c, b, stobj+10);	
			break;
			case VPB8L:					
				// PACKED_FIFO_UP
				config_reset_fifo_alarm(vpb_c, b, stobj+6);
			break;
			default:
				throw Wobbly(VPBAPI_MODEL_NOT_SUPPORTED);
			break;
		}
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_reset_record_underflow"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: vpb_reset_play_fifo_alarm
	AUTHOR..: David Rowe
	DATE....: 23/9/98

	Resets the latch that detects record DSP buffer underflows.  After
	calling this function another event may be generated if the buffer
	underflows.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_reset_play_fifo_alarm(int handle)
{
	USHORT	stobj,b,ch;

	try {
		ValidHandleCheck(handle);
		maphndletodev(handle, &b, &ch);
		stobj = objtrack_handle_to_id(DNOBJ, handle);

		assert(vpb_c->vpbreg(b)->model==VPB4);
		config_reset_fifo_alarm(vpb_c, b, stobj);  // PACKED_FIFO_UP
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_reset_record_underflow"));
	}
	
	return(VPB_OK);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: playrec_underflow_valid
	AUTHOR..: David Rowe
	DATE....: 19/10/98

	Returns non-zero if an underflow is valid.  At certain times (like
	when we are waiting for the FIFO to empty at the end of a play
	session), underflow events from the DSP are not valid.

\*--------------------------------------------------------------------------*/

int playrec_underflow_valid(int handle) {
	return(play[handle].underflow_valid);
}

/*-------------------------------------------------------------------------*\
        
        FUNCTION: gain_vector
	AUTHOR..: John Kostogiannis
	DATE....: 09/04/02
       
        Multiplies a vector by a gain parameter. Also includes hard limiter.

\*-------------------------------------------------------------------------*/

int gain_vector(float g, short *v, int n) {
        int i;
	float tmp;
	for ( i = 0; i<n; i++) {
	    tmp = g*v[i];
	    if (tmp > 32767.0)
	      tmp = 32767.0;
	    if (tmp < -32768.0)
	      tmp = -32768.0;
	    v[i] = (short)tmp;	
	}  
	return(i);
}	  
	  
/*-------------------------------------------------------------------------*\
        
        FUNCTION: hp_filter
	AUTHOR..: David Rowe
	DATE....: 12/9/02
       
        Applies a 2nd order HP filter to a vector of samples.

\*-------------------------------------------------------------------------*/

void hp_filter(float states[HP_ORDER-1], float coeff[HP_ORDER],  
		short *v, int n) {
  int i;
  float x,y;

  for(i=0; i<n; i++) {
	  x = (float)v[i];
	  y = x*coeff[0] + states[0]*coeff[1] + states[1]*coeff[2];
	  states[1] = states[0];
	  states[0] = x;
	  if (y > 32767.0)
		  y = 32767.0;
	  if (y < -32768.0)
		  y = -32768.0;
	  v[i] = (short)y;	
  }

}






